% This script reads ADMD data and GXP forecast data and produces future
% grid loading cases, ready for loading into powerflow software. Data can
% be written out in either IDEV form for PSS/E, or DLE form for DigSilent.

% Bruce Smith, Electricity Commission

% Amended by Brian Kirtlan 23/11/2005 to handle and output island/regional data
% Last amended Brian Kirtlan January 2008 

% The order of the adjustments to the forecasts has been changed in this version of the script to correct the point at which 
% the scaling of the raw peak forecasts to the prudent peak forecasts occurs 

% The script follows these high level steps:
% 1. Load data and overwrite if specified by switch settings
% 2. Adjust regional prudent peak forecasts so that they increase at the mean rate of growth after year 5
% 3. Map GWh forecasts from the demand forecast GXP definitions to the ADMD definitions and consolidate if required
% 4. Calculate base GXP peak demand forecasts
% 5. Make adjustments to the peak forecasts and diversity factors to reflect any new loads
% 6. Calculate regional scaling factors and apply
% 7. Accumulate region and island after diversity totals and calculate power factors
% 8. Build PSSE mapping tables
% 9. Write PSSE files
% 10. Write ADMD .csv file

clear all
tic

originaldirectory = cd;
InputDir = 'C:\\NatDemand2009\\ADMD\\';
cd(InputDir);

% Comment next two lines out if don't need to run Base ADMD script
%BaseADMD
%clear all


% Model switches

ExportDataType = 1; % Set 1 = GXP loads at island peak, 2 = GXP loads at island minimum, 3 = GXP loads at region peak
AdjustforNewLoads = 1; % Set to 1 to include specified load adjustments in forecasts (should be consistent with regional totals from Regional Energy Model passed to calculate MaxLikelyPeaks)
AddBackEmbedded = 1; % 1 = adjust forecasts to include estimated growth from load supplied by embedded generation, any other value = don't adjust   
MaxPeakScalingHorizon = 5; % Number of years after which the scaling applied between the MaxLikelyPeak and the MeanPeak are fixed (i.e. all years after year X are scaled at the same ratio as year X 
ScaletoMaxLikelihoodPeaks = 1; % scale GXP peak values so that the region peaks match the max likelihood region peaks
MoveMarchYearBack = 1; % scales the energy forecast March year backwards rather than forward (March year ended 07 is treated as though its the 2006 calendar year for the purposes of calculating growth rates to apply to peaks) 
KeepTiwaiPeakConstant = 1; % keeps Tiwai constant during scaling of regional demand to the 10% POE figures (extracts before calculating scaling - causes a small amount of distortion upwards(about 4MW in 2040))
SubMeanforPrudentPeak = 0; % substitute mean regional forecasts for the prudent peak regional forecast so that producing mean GXP forecasts  
UseFixedPFs = 0;
FixedPF = 0.9500;

PSSE_NorthIsland_File='NAaN Peak Winter_2007.raw'; % file to use for importing North Island PSSE bus numbers from
PSSE_SouthIsland_File='SI Winter 2007.raw'; % file to use for importing South Island PSSE bus numbers from


warning off MATLAB:divideByZero

disp('Starting ADMD forecasts and outputting PSS/E & DigSilent files') 


load ADMDdata % load diverity and power factor results (tables created by BaseADMD.m script)
load 2009GXPForecasts % load GWh GXPForecast which is output from the regional energy forecast model.  Regions is loaded from here
load 2009ADMDInputs % mapping, peak forecasts
load 2009PFandDFOverwrites ADMDDFandPFOverwrites % overwrite table for PF and DF values where the metered data gives undesirable (nonsensical) results

if SubMeanforPrudentPeak == 1
    MaxLikelyPeak = MeanPeak;
end

YearStt=2009; %the desired range of output load cases
YearStp=2044;
nGXP=size(BasePeak,2);
nRegions = length(Regions);

GXPForecastYears = ForecastYears';
nn = length(GXPForecastYears);

OtagoSouthRegion = strmatch('Otago/Southland',Regions);
TiwaiGXP = strmatch('TWI2201',ADMDGXPMapping(:,2));
if isempty(OtagoSouthRegion); error('WARNING - NO MATCH TO OTAGO/SOUTHLAND IN REGIONS TABLE'); end;
if isempty(TiwaiGXP); error('WARNING - NO MATCH TO TWI220 IN ADMD GXP MAPPING TABLE'); end;
TiwaiPeak = GXPPeakDemand(TiwaiGXP)./1000;

% Make any PF and DF overwrites

nOverwrites = size(ADMDDFandPFOverwrites,1);
for gxp = 1:nOverwrites
    GXPidx = strmatch(ADMDDFandPFOverwrites(gxp,1),ADMDPSSEMapping(:,2));
    if ~isempty(ADMDDFandPFOverwrites{gxp,2}); FYRegionPF(GXPidx)=ADMDDFandPFOverwrites{gxp,2};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,3}); FYIslandPF(GXPidx)=ADMDDFandPFOverwrites{gxp,3};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,4}); WIslandPF(GXPidx)=ADMDDFandPFOverwrites{gxp,4};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,5}); SIslandPF(GXPidx)=ADMDDFandPFOverwrites{gxp,5};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,6}); FYRegionDF(GXPidx)=ADMDDFandPFOverwrites{gxp,6};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,7}); FYIslandDF(GXPidx)=ADMDDFandPFOverwrites{gxp,7};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,8}); WIslandDF(GXPidx)=ADMDDFandPFOverwrites{gxp,8};end
    if ~isempty(ADMDDFandPFOverwrites{gxp,9}); SIslandDF(GXPidx)=ADMDDFandPFOverwrites{gxp,9};end
end;

% set all power factors to a constant if required

if UseFixedPFs == 1;
    ESIslandPF(:,:) = FixedPF; 
    SIslandPF(:,:) = FixedPF;
    WIslandPF(:,:) = FixedPF;
    FYIslandPF(:,:) = FixedPF;
    ESRegionPF(:,:) = FixedPF;
    SRegionPF(:,:) = FixedPF;
    WRegionPF(:,:) = FixedPF;
    FYRegionPF(:,:) = FixedPF;
    ESIslandLightLoadPF(:,:) = FixedPF;
    SIslandLightLoadPF(:,:) = FixedPF;
    WIslandLightLoadPF(:,:) = FixedPF;
    FYIslandLightLoadPF (:,:) = FixedPF;  
end;

% check year range is consistent with input data

if BaseYear < (GXPForecastYears(1))
    disp('First year of forecast does not reach back to the BaseYear, quiting')
    return
end
if YearStt < BaseYear
    disp('First year of forecast is before the BaseYear, quiting')
    return
end
if YearStp > (GXPForecastYears(nn))
    disp('Last year of forecast is beyond the supplied forecast data, quiting')
    return
end

% set up adjusted peak forecasts

PeakScaleHorizon = MaxPeakScalingHorizon + YearStt;
HorizonYrIdx = find(PeakYear==PeakScaleHorizon) + 1; %note that 2011 ADMD = 2012 year ending March from the POE forecasts

% and take out Tiwai if keeping constant
if KeepTiwaiPeakConstant == 1
    MaxLikelyPeak(1:length(MaxLikelyPeak),OtagoSouthRegion) = MaxLikelyPeak(1:length(MaxLikelyPeak),OtagoSouthRegion) - TiwaiPeak; % 10 = 2007
    MeanPeak(1:length(MeanPeak),OtagoSouthRegion) = MeanPeak(1:length(MeanPeak),OtagoSouthRegion) - TiwaiPeak; % 11 = 2007 Calendar
end    

RawPeakRatio = MaxLikelyPeak./MeanPeak;
AdjMaxLikelyPeak(1:HorizonYrIdx,:)=MeanPeak(1:HorizonYrIdx,:).*RawPeakRatio(1:HorizonYrIdx,:);
for reg = 1:nRegions;
    AdjMaxLikelyPeak(HorizonYrIdx+1:length(PeakYear),reg)=MeanPeak(HorizonYrIdx+1:length(PeakYear),reg).*RawPeakRatio(HorizonYrIdx,reg);
end;

% Map the GXP forecast data to the ADMD names

% If adding in new loads use the adjusted GXP energy forecasts
if AdjustforNewLoads == 1
    GXPTotalForecastDemand = GXPTotalForecastDemandAdj;    
end;

% Consolidate GXPs where necessary (defined in the GXPConsolidation - first column = original GXP, second column = target GXP to consolidate any data to)

for gxp = 1:(size(GXPConsolidationList,1))
    SourceGXP = GXPConsolidationList{gxp,1};
    TargetGXP = GXPConsolidationList{gxp,2};
    GXPSourceRef = strmatch(SourceGXP,GXPForecastNames);
    GXPTargetRef = strmatch(TargetGXP,GXPForecastNames);
    GXPTotalForecastDemand(:,GXPTargetRef) = GXPTotalForecastDemand(:,GXPTargetRef)+GXPTotalForecastDemand(:,GXPSourceRef);    
    GXPTotalForecastDemand(:,GXPSourceRef) = 0;
    GXPExistingEmbeddedMW(GXPTargetRef) = GXPExistingEmbeddedMW(GXPTargetRef)+GXPExistingEmbeddedMW(GXPSourceRef);
    GXPExistingEmbeddedMW(GXPSourceRef) = 0;
end;

% Map to the PSSE (ADMD) list of GXP names

ADMDGXPForecastRates = zeros(nn,nGXP);
ADMDGXPExistingEmbeddedMW = zeros(1,nGXP);
ADMDIsland = cell(nGXP,1);

for gxp = 1:nGXP
    GXPMappingRef = strmatch(ADMDnames(gxp), ADMDGXPMapping(:,1), 'exact');
    GXPSource = ADMDGXPMapping(GXPMappingRef,2);
    ADMDIsland(gxp) = ADMDGXPMapping(GXPMappingRef,3);
    if not(strcmp(GXPSource{1},'NOGXP'))
        GXPSourceRef = strmatch(GXPSource, GXPForecastNames);
        ADMDGXPForecastRates(:,gxp) = GXPTotalForecastDemand(:,GXPSourceRef);% note that the ADMDGXPForecastRates matix indicates growth rates as GWh. Some source GXPs are repeated in more than one target GXP
        ADMDGXPExistingEmbeddedMW(1,gxp) = GXPExistingEmbeddedMW(GXPSourceRef);
    end;
end;
GXPForecastRatio = zeros(nn,nGXP);
GXPForecastRatio(1,:) = 1;
for yr = 1:nn-1
    for gxp = 1:nGXP
        if ADMDGXPForecastRates(1,gxp) == 0
            GXPForecastRatio(yr+1,gxp) = 1;
        else
            GXPForecastRatio(yr+1,gxp) = ADMDGXPForecastRates(yr+1,gxp)./ADMDGXPForecastRates(1,gxp);
        end;
    end;
end;

% Map NewLoad GXPs to consolidated GXPs if required
if exist('NewLoad')==1 && not(isempty(NewLoad{1,1}));
    NewLoadDim = size(NewLoad);
    for NewLoadNum = 1:NewLoadDim(1);
        if not(isempty(strmatch(NewLoad{NewLoadNum,2}, GXPConsolidationList(:,1), 'exact')))
           NewLoad{NewLoadNum,2} = GXPConsolidationList(strmatch(NewLoad{NewLoadNum,2}, GXPConsolidationList(:,1), 'exact'),2);
        end;    
    end; 
end;

% Set any diversity factor and power factor NaN values to 1

ESPF(isnan(ESPF))=1;
ESDF(isnan(ESDF))=1;
SPF(isnan(SPF))=1;
SDF(isnan(SDF))=1;
WPF(isnan(WPF))=1;
WDF(isnan(WDF))=1;
FYPF(isnan(FYPF))=1;
FYDF(isnan(FYDF))=1;

ESIslandPF(isnan(ESIslandPF))=1;
ESIslandDF(isnan(ESIslandDF))=1;
SIslandPF(isnan(SIslandPF))=1;
SIslandDF(isnan(SIslandDF))=1;
WIslandPF(isnan(WIslandPF))=1;
WIslandDF(isnan(WIslandDF))=1;
FYIslandPF(isnan(FYIslandPF))=1;
FYIslandDF(isnan(FYIslandDF))=1;
ESRegionPF(isnan(ESRegionPF))=1;
ESRegionDF(isnan(ESRegionDF))=1;
SRegionPF(isnan(SRegionPF))=1;
SRegionDF(isnan(SRegionDF))=1;
WRegionPF(isnan(WRegionPF))=1;
WRegionDF(isnan(WRegionDF))=1;
FYRegionPF(isnan(FYRegionPF))=1;
FYRegionDF(isnan(FYRegionDF))=1;
ESIslandLightLoadPF(isnan(ESIslandLightLoadPF))=1;
ESIslandLightLoadDF(isnan(ESIslandLightLoadDF))=1;
SIslandLightLoadPF(isnan(SIslandLightLoadPF))=1;
SIslandLightLoadDF(isnan(SIslandLightLoadDF))=1;
WIslandLightLoadPF(isnan(WIslandLightLoadPF))=1;
WIslandLightLoadDF(isnan(WIslandLightLoadDF))=1;
FYIslandLightLoadPF(isnan(FYIslandLightLoadPF))=1;
FYIslandLightLoadDF(isnan(FYIslandLightLoadDF))=1;

% Create a GXP region table

GXPReg = zeros(1,nGXP);
for ii = 1:nGXP;
    GXPReg(ii)= strmatch(ADMDGXPMapping(strmatch(ADMDnames(ii),ADMDGXPMapping(:,1),'exact'),4),Regions); 
end;

% Calculate pre-scaled GXP and regional forecasts

GXPForecastP = zeros(nGXP,nn);

if GXPForecastYears(1)==(BaseYear+1)
    BaseIdx=0;
else
    BaseIdx=find(GXPForecastYears==(BaseYear+1))-1;
end

for ii=1:nGXP
    
    GXPForecastP(ii,1)= BasePeak(ii)./1000;
   
    % Project MW forecasts based on GXP growth ratios from GWh forecasts
    
    for jj=2:nn
        if isnan(GXPForecastRatio(BaseIdx+jj-1,ii)) | (KeepTiwaiPeakConstant == 1 & ii==TiwaiGXP) ;
            gf=1;
        else
            gf=GXPForecastRatio(BaseIdx+jj-1,ii);
        end;
        GXPForecastP(ii,jj)=GXPForecastP(ii,1)*gf + (AddBackEmbedded ==1).* (ADMDGXPExistingEmbeddedMW(ii).*(gf-1));
    end

    % Change any NaNs to 0
    
    GXPForecastP(ii,isnan(GXPForecastP(ii,:)))=0;    
    
end;  

% Adjust forecasts for any assumed new loads

if AdjustforNewLoads == 1   
    
    % Known major new loads are specified in the "NewLoad" table included in the in the Matlab workspace used for the regional demand forecasts

    if exist('NewLoad')==1 && not(isempty(NewLoad{1,1}));
        for NewLoadNum = 1:NewLoadDim(1);
            NewLoadBusName = NewLoad{NewLoadNum,2};
            NewLoadMWRaw = NewLoad{NewLoadNum,3};          
            NewLoadGWh = NewLoad{NewLoadNum,4};
            NewLoadStartYear = NewLoad{NewLoadNum,5};
            NewLoadEndYear = NewLoad{NewLoadNum,6};
            NewLoadAdditiveFlag = NewLoad{NewLoadNum,7};
            NewLoadGXPDiv = NewLoad{NewLoadNum,9};
            NewLoadESRegDiv = NewLoad{NewLoadNum,10};
            NewLoadSRegDiv = NewLoad{NewLoadNum,11};
            NewLoadWRegDiv = NewLoad{NewLoadNum,12};
            NewLoadFYRegDiv = NewLoad{NewLoadNum,13};
            NewLoadESIsDiv = NewLoad{NewLoadNum,14};
            NewLoadSIsDiv = NewLoad{NewLoadNum,15};
            NewLoadWIsDiv = NewLoad{NewLoadNum,16};
            NewLoadFYIsDiv = NewLoad{NewLoadNum,17};
            NewLoadESIsMinDiv = NewLoad{NewLoadNum,18};
            NewLoadSIsMinDiv = NewLoad{NewLoadNum,19};
            NewLoadWIsMinDiv = NewLoad{NewLoadNum,20};
            NewLoadFYIsMinDiv = NewLoad{NewLoadNum,21};
            AdjGXP = strmatch(NewLoadBusName,ADMDGXPMapping(:,2));
%AK
            if ADMDGXPMapping{AdjGXP,3} == char('N') % translate island reference to numbers
                IslandRef = 1;
            else
                IslandRef = 2;
            end;
              
%            IslandRef = 2-(ADMDGXPMapping{AdjGXP,3} == char('N')); % North is row 1, South is row 2

            RegionRef = strmatch(ADMDGXPMapping(AdjGXP,4),Regions);            
            % Calculate load adjustments for the GXP series and the resulting impact on Regional and Island diversity 
            % The MW adj needs to ignore growth already reflected in the increase in GWh 
            % Peak growth in GWh = MW*8760hrs/1000*AverageLoadProportion. AverageLoadProportion = GWh/(8.76*MW). MW already included = MW(Original)*GWH(NewLoad)/GWh(Original). Adjustment = MW * (new load diversity - AverageLoadProp(new)/AverageLoadProp(old))
            % The Island and Regional totals need to be individually adjusted to reflect the selected diversity for the new load for each total type
            OrigLoadGwh = GXPTotalForecastDemand(1,strmatch(ADMDGXPMapping(AdjGXP,2),GXPForecastNames)); 
            OrigLoadMW = GXPForecastP(AdjGXP,1);
            if OrigLoadGwh == 0 % catches cases where initial load is 0 Gwh 
                DiversityAdj = 0;
            else
                DiversityAdj = (NewLoadGWh./(8.76.*NewLoadMWRaw))./((OrigLoadGwh./(8.76.*OrigLoadMW)));
            end;
            NewLoadMWGXP = NewLoadMWRaw .* (NewLoadGXPDiv - DiversityAdj); % backing out impact of peak adjustments included in the GWh demand forecast 
               
            % Revise load factors to account for new load (ideally we would have varying load factors across time and / or projections of load duration curves)
            % new combined diversity = (New Load Diversity*NewLoad + OldLoadDiversity*OldLoad(prenewloadincludedinGWhforecast) / (Oldload + New Load)
            
            ESDF(AdjGXP) = (NewLoadESIsDiv .* NewLoadMWRaw + ESDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            SDF(AdjGXP) = (NewLoadSIsDiv .* NewLoadMWRaw + SDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            WDF(AdjGXP) = (NewLoadWIsDiv .* NewLoadMWRaw + WDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            FYDF(AdjGXP) = (NewLoadFYIsDiv .* NewLoadMWRaw + FYDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            ESRegionDF(AdjGXP) = (NewLoadESRegDiv .* NewLoadMWRaw + ESRegionDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            SRegionDF(AdjGXP) = (NewLoadSRegDiv .* NewLoadMWRaw + SRegionDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            WRegionDF(AdjGXP) = (NewLoadWRegDiv .* NewLoadMWRaw + WRegionDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            FYRegionDF(AdjGXP) = (NewLoadFYRegDiv .* NewLoadMWRaw + FYRegionDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            ESIslandDF(AdjGXP) = (NewLoadESIsDiv .* NewLoadMWRaw + ESIslandDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            SIslandDF(AdjGXP) = (NewLoadSIsDiv .* NewLoadMWRaw + SIslandDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            WIslandDF(AdjGXP) = (NewLoadWIsDiv .* NewLoadMWRaw + WIslandDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            FYIslandDF(AdjGXP) = (NewLoadFYIsDiv .* NewLoadMWRaw + FYIslandDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            ESIslandLightLoadDF(AdjGXP) = (NewLoadESIsMinDiv .* NewLoadMWRaw + ESIslandLightLoadDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            SIslandLightLoadDF(AdjGXP) = (NewLoadSIsMinDiv .* NewLoadMWRaw + SIslandLightLoadDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            WIslandLightLoadDF(AdjGXP) = (NewLoadWIsMinDiv .* NewLoadMWRaw + WIslandLightLoadDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            FYIslandLightLoadDF(AdjGXP) = (NewLoadFYIsMinDiv .* NewLoadMWRaw + FYIslandLightLoadDF(AdjGXP).*OrigLoadMW)./(NewLoadMWRaw+OrigLoadMW);
            
            % adjust GXP load
            if not(isempty(AdjGXP))
                if NewLoadStartYear < GXPForecastYears(1)
                    NewLoadStartYear = GXPForecastYears(1);
                    disp(['Warning - new load start date is earlier than the first forecast period for ' NewLoad{NewLoadNum,1}]);
                end;
                if NewLoadEndYear > GXPForecastYears(1)+nn-1;
                    NewLoadEndYear = GXPForecastYears(1)+nn-1;
                end;
                if NewLoadEndYear>=NewLoadStartYear;
                    for AdjYear = 1:nn;
                        if (AdjYear + YearStt -1 >= NewLoadStartYear) & (AdjYear + YearStt - 1 <= NewLoadEndYear)
                            % Add load to the specified bus and to the various totals                        
                            if AddBackEmbedded ~= 1 
                                GXPForecastP(AdjGXP,AdjYear) = GXPForecastP(AdjGXP,AdjYear) + NewLoadMWGXP;
                            else
                                % if also adjusting for embedded gen need to back out the effect of increasing the 'hidden' demand supplied by embedded gen at the same ratio as at the GXP (which includes the new load)
                                % the 'over-adjustment' that needs to be backed out is EmbeddedGenMW x the ratio of NewLoadGWh/BaseGWH  
                                if ADMDGXPForecastRates(1,AdjGXP) == 0
                                    GXPForecastP(AdjGXP,AdjYear) = GXPForecastP(AdjGXP,AdjYear) + NewLoadMWGXP; % new GXPs with no existing load
                                else
                                    GXPForecastP(AdjGXP,AdjYear) = GXPForecastP(AdjGXP,AdjYear) + NewLoadMWGXP - (ADMDGXPExistingEmbeddedMW(AdjGXP).*(NewLoadGWh./ADMDGXPForecastRates(1,AdjGXP)));                           
                                end;                                                         
                            end;                        
                            % If not additive, spread and subtract the load from all other buses
                            if NewLoadAdditiveFlag ~= 1;
                                
                                % need to calculate ratios etc for other GXPS
                                % NOT USED - NOT CURRENTLY CODED
                                
                            end;                   
                        end;                    
                    end;                    
                else
                    disp(['Warning - new load start date is later than the new load end date for ' NewLoad{NewLoadNum,1}]);
                end;
                
            else
                disp(['Warning - new load bus name does not match existing bus names for ' NewLoad{NewLoadNum,1}]);
            end;
        end;
    end;
end; % endif AdjustforNewLoads=1   


% Apply scaling factors if required
% Scaling is applied across all GXPs within the region except Tiwai if the option to exclude it is selected
if ScaletoMaxLikelihoodPeaks == 1

    PeakYrRegionAdjFactor = zeros(nRegions,nn);
    RawRegionTotal = zeros(nRegions,nn);
    GXPRegAdj = zeros(1,nn);

    for ii=1:nGXP
        RawRegionTotal(GXPReg(ii),:) = RawRegionTotal(GXPReg(ii),:) + GXPForecastP(ii,:).*FYRegionDF(ii);
    end
    if KeepTiwaiPeakConstant == 1
        RawRegionTotal(OtagoSouthRegion,:)=RawRegionTotal(OtagoSouthRegion,:)-TiwaiPeak;
    end;
    if MoveMarchYearBack ==1
        YrOffset = YearStt-PeakYear(1)+1;
    else
        YrOffset = YearStt-PeakYear(1);
    end;
    
    for reg = 1:nRegions
        for AdjYear = 1:nn-1
            PeakYrRegionAdjFactor(reg,AdjYear) = AdjMaxLikelyPeak(AdjYear+YrOffset,reg)'./RawRegionTotal(reg,AdjYear);         
        end;
    end;

    for ii=1:nGXP
        if ii~=TiwaiGXP | KeepTiwaiPeakConstant ~= 1
            for AdjYear = 1:nn-1
                GXPForecastP(ii,AdjYear) = GXPForecastP(ii,AdjYear).*PeakYrRegionAdjFactor(GXPReg(ii),AdjYear);
            end;
        end;
    end;
end;

 

% Calculate reactive power and accumulate region and island totals  

% Power factors

% Pre-declare variables

ESForecastP=zeros(nGXP,nn); % GXP peaks
SForecastP=zeros(nGXP,nn);
WForecastP=zeros(nGXP,nn);
FYForecastP=zeros(nGXP,nn);

ESIslandForecastP=zeros(nGXP,nn); % GXP load at Island peak
SIslandForecastP=zeros(nGXP,nn);
WIslandForecastP=zeros(nGXP,nn);
FYIslandForecastP=zeros(nGXP,nn);

ESRegionForecastP=zeros(nGXP,nn);% GXP load at Region peak
SRegionForecastP=zeros(nGXP,nn);
WRegionForecastP=zeros(nGXP,nn);
FYRegionForecastP=zeros(nGXP,nn);

ESIslandLightForecastP=zeros(nGXP,nn);% GXP load at Island minimum
SIslandLightForecastP=zeros(nGXP,nn);
WIslandLightForecastP=zeros(nGXP,nn);
FYIslandLightForecastP=zeros(nGXP,nn);


ESRegionIslandPeakForecast = zeros(nRegions,nn); % Region load at Island peak
SRegionIslandPeakForecast = zeros(nRegions,nn);
WRegionIslandPeakForecast = zeros(nRegions,nn);
FYRegionIslandPeakForecast = zeros(nRegions,nn);

ESRegionForecast = zeros(nRegions,nn); % Region peak
SRegionForecast = zeros(nRegions,nn);
WRegionForecast = zeros(nRegions,nn);
FYRegionForecast = zeros(nRegions,nn);

ESRegionIslandLightForecast = zeros(nRegions,nn); % Region load at Island minimum 
SRegionIslandLightForecast = zeros(nRegions,nn);
WRegionIslandLightForecast = zeros(nRegions,nn);
FYRegionIslandLightForecast = zeros(nRegions,nn);

ESIslandForecast = zeros(2,nn); % Island peak
SIslandForecast = zeros(2,nn);
WIslandForecast = zeros(2,nn);
FYIslandForecast = zeros(2,nn);

ESIslandLightForecast = zeros(2,nn); % Island minimum
SIslandLightForecast = zeros(2,nn);
WIslandLightForecast = zeros(2,nn);
FYIslandLightForecast = zeros(2,nn);

ESNationalForecast = zeros(1,nn); % National peak
SNationalForecast = zeros(1,nn);
WNationalForecast = zeros(1,nn);
FYNationalForecast = zeros(1,nn);

ESIslandNationalForecast = zeros(2,nn); % Island totals at time of National peak
SIslandNationalForecast = zeros(2,nn);
WIslandNationalForecast = zeros(2,nn);
FYIslandNationalForecast = zeros(2,nn);


ESForecastQ=zeros(nGXP,nn);
SForecastQ=zeros(nGXP,nn);
WForecastQ=zeros(nGXP,nn);
FYForecastQ=zeros(nGXP,nn);

ESIslandForecastQ=zeros(nGXP,nn);
SIslandForecastQ=zeros(nGXP,nn);
WIslandForecastQ=zeros(nGXP,nn);
FYIslandForecastQ=zeros(nGXP,nn);

ESRegionForecastQ=zeros(nGXP,nn);
SRegionForecastQ=zeros(nGXP,nn);
WRegionForecastQ=zeros(nGXP,nn);
FYRegionForecastQ=zeros(nGXP,nn);

ESIslandLightForecastQ=zeros(nGXP,nn);
SIslandLightForecastQ=zeros(nGXP,nn);
WIslandLightForecastQ=zeros(nGXP,nn);
FYIslandLightForecastQ=zeros(nGXP,nn);


for ii=1:nGXP   
    
    % Calculate GXP loads at Island and Regional peaks
    
    % GXP load at national peak
    ESForecastP(ii,:)=GXPForecastP(ii,:).*ESDF(ii);
    SForecastP(ii,:)=GXPForecastP(ii,:).*SDF(ii);
    WForecastP(ii,:)=GXPForecastP(ii,:).*WDF(ii);
    FYForecastP(ii,:)=GXPForecastP(ii,:).*FYDF(ii);
    % GXP load at island peak
    ESIslandForecastP(ii,:)=GXPForecastP(ii,:).*ESIslandDF(ii);
    SIslandForecastP(ii,:)=GXPForecastP(ii,:).*SIslandDF(ii);
    WIslandForecastP(ii,:)=GXPForecastP(ii,:).*WIslandDF(ii);
    FYIslandForecastP(ii,:)=GXPForecastP(ii,:).*FYIslandDF(ii);
    % GXP load at region peak
    ESRegionForecastP(ii,:)=GXPForecastP(ii,:).*ESRegionDF(ii);
    SRegionForecastP(ii,:)=GXPForecastP(ii,:).*SRegionDF(ii);
    WRegionForecastP(ii,:)=GXPForecastP(ii,:).*WRegionDF(ii);
    FYRegionForecastP(ii,:)=GXPForecastP(ii,:).*FYRegionDF(ii);
    % GXP load at island minimum (light load)
    ESIslandLightForecastP(ii,:)=GXPForecastP(ii,:).*ESIslandLightLoadDF(ii);
    SIslandLightForecastP(ii,:)=GXPForecastP(ii,:).*SIslandLightLoadDF(ii);
    WIslandLightForecastP(ii,:)=GXPForecastP(ii,:).*WIslandLightLoadDF(ii);
    FYIslandLightForecastP(ii,:)=GXPForecastP(ii,:).*FYIslandLightLoadDF(ii);
    
    % Change any NaNs to 0
    
    % national peak
    ESForecastP(ii,isnan(ESForecastP(ii,:)))=0;
    SForecastP(ii,isnan(SForecastP(ii,:)))=0;
    WForecastP(ii,isnan(WForecastP(ii,:)))=0;
    FYForecastP(ii,isnan(FYForecastP(ii,:)))=0;
    % island peak
    ESIslandForecastP(ii,isnan(ESIslandForecastP(ii,:)))=0;
    SIslandForecastP(ii,isnan(SIslandForecastP(ii,:)))=0;
    WIslandForecastP(ii,isnan(WIslandForecastP(ii,:)))=0;
    FYIslandForecastP(ii,isnan(FYIslandForecastP(ii,:)))=0;
    % region peak
    ESRegionForecastP(ii,isnan(ESRegionForecastP(ii,:)))=0;
    SRegionForecastP(ii,isnan(SRegionForecastP(ii,:)))=0;
    WRegionForecastP(ii,isnan(WRegionForecastP(ii,:)))=0;
    FYRegionForecastP(ii,isnan(FYRegionForecastP(ii,:)))=0;
    % island minimum
    ESIslandLightForecastP(ii,isnan(ESIslandLightForecastP(ii,:)))=0;
    SIslandLightForecastP(ii,isnan(SIslandLightForecastP(ii,:)))=0;
    WIslandLightForecastP(ii,isnan(WIslandLightForecastP(ii,:)))=0;
    FYIslandLightForecastP(ii,isnan(FYIslandLightForecastP(ii,:)))=0;
    
    % Calculate reactive power 
    
    % national peak
    ESForecastQ(ii,1)=tan(acos(ESPF(ii))).*ESForecastP(ii,1); %ignore power angle data, so Q is always positive...
    SForecastQ(ii,1)=tan(acos(SPF(ii))).*SForecastP(ii,1);    
    WForecastQ(ii,1)=tan(acos(WPF(ii))).*WForecastP(ii,1);    
    FYForecastQ(ii,1)=tan(acos(FYPF(ii))).*FYForecastP(ii,1);    
    % island peak
    ESIslandForecastQ(ii,1)=tan(acos(ESIslandPF(ii))).*ESIslandForecastP(ii,1);
    SIslandForecastQ(ii,1)=tan(acos(SIslandPF(ii))).*SIslandForecastP(ii,1);
    WIslandForecastQ(ii,1)=tan(acos(WIslandPF(ii))).*WIslandForecastP(ii,1);
    FYIslandForecastQ(ii,1)=tan(acos(FYIslandPF(ii))).*FYIslandForecastP(ii,1);
    % region peak
    ESRegionForecastQ(ii,1)=tan(acos(ESRegionPF(ii))).*ESRegionForecastP(ii,1);
    SRegionForecastQ(ii,1)=tan(acos(SRegionPF(ii))).*SRegionForecastP(ii,1);
    WRegionForecastQ(ii,1)=tan(acos(WRegionPF(ii))).*WRegionForecastP(ii,1);
    FYRegionForecastQ(ii,1)=tan(acos(FYRegionPF(ii))).*FYRegionForecastP(ii,1);
    % island minimum (light load)
    ESIslandLightForecastQ(ii,1)=tan(acos(ESIslandLightLoadPF(ii))).*ESIslandLightForecastP(ii,1);
    SIslandLightForecastQ(ii,1)=tan(acos(SIslandLightLoadPF(ii))).*SIslandLightForecastP(ii,1);
    WIslandLightForecastQ(ii,1)=tan(acos(WIslandLightLoadPF(ii))).*WIslandLightForecastP(ii,1);
    FYIslandLightForecastQ(ii,1)=tan(acos(FYIslandLightLoadPF(ii))).*FYIslandLightForecastP(ii,1);  
    
    for jj=2:nn
        % national peak
        ESForecastQ(ii,jj)=tan(acos(ESPF(ii))).*ESForecastP(ii,jj);
        SForecastQ(ii,jj)=tan(acos(SPF(ii))).*SForecastP(ii,jj);
        WForecastQ(ii,jj)=tan(acos(WPF(ii))).*WForecastP(ii,jj);
        FYForecastQ(ii,jj)=tan(acos(FYPF(ii))).*FYForecastP(ii,jj);
        % island peak
        ESIslandForecastQ(ii,jj)=tan(acos(ESIslandPF(ii))).*ESIslandForecastP(ii,jj);
        SIslandForecastQ(ii,jj)=tan(acos(SIslandPF(ii))).*SIslandForecastP(ii,jj);
        WIslandForecastQ(ii,jj)=tan(acos(WIslandPF(ii))).*WIslandForecastP(ii,jj);
        FYIslandForecastQ(ii,jj)=tan(acos(FYIslandPF(ii))).*FYIslandForecastP(ii,jj);    
        % region peak
        ESRegionForecastQ(ii,jj)=tan(acos(ESRegionPF(ii))).*ESRegionForecastP(ii,jj);
        SRegionForecastQ(ii,jj)=tan(acos(SRegionPF(ii))).*SRegionForecastP(ii,jj);
        WRegionForecastQ(ii,jj)=tan(acos(WRegionPF(ii))).*WRegionForecastP(ii,jj);
        FYRegionForecastQ(ii,jj)=tan(acos(FYRegionPF(ii))).*FYRegionForecastP(ii,jj);
        % island minimum
        ESIslandLightForecastQ(ii,jj)=tan(acos(ESIslandLightLoadPF(ii))).*ESIslandLightForecastP(ii,jj);
        SIslandLightForecastQ(ii,jj)=tan(acos(SIslandLightLoadPF(ii))).*SIslandLightForecastP(ii,jj);
        WIslandLightForecastQ(ii,jj)=tan(acos(WIslandLightLoadPF(ii))).*WIslandLightForecastP(ii,jj);
        FYIslandLightForecastQ(ii,jj)=tan(acos(FYIslandLightLoadPF(ii))).*FYIslandLightForecastP(ii,jj);  
    end
    
    % Change any NaNs to 0
    
    % national peak
    ESForecastQ(ii,isnan(ESForecastQ(ii,:)))=0;
    SForecastQ(ii,isnan(SForecastQ(ii,:)))=0;
    WForecastQ(ii,isnan(WForecastQ(ii,:)))=0;
    FYForecastQ(ii,isnan(FYForecastQ(ii,:)))=0;
    % island peak
    ESIslandForecastQ(ii,isnan(ESIslandForecastQ(ii,:)))=0;
    SIslandForecastQ(ii,isnan(SIslandForecastQ(ii,:)))=0;
    WIslandForecastQ(ii,isnan(WIslandForecastQ(ii,:)))=0;
    FYIslandForecastQ(ii,isnan(FYIslandForecastQ(ii,:)))=0;
    % region peak
    ESRegionForecastQ(ii,isnan(ESRegionForecastQ(ii,:)))=0;
    SRegionForecastQ(ii,isnan(SRegionForecastQ(ii,:)))=0;
    WRegionForecastQ(ii,isnan(WRegionForecastQ(ii,:)))=0;
    FYRegionForecastQ(ii,isnan(FYRegionForecastQ(ii,:)))=0;
    % island minimum
    ESIslandLightForecastQ(ii,isnan(ESIslandLightForecastQ(ii,:)))=0;
    SIslandLightForecastQ(ii,isnan(SIslandLightForecastQ(ii,:)))=0;
    WIslandLightForecastQ(ii,isnan(WIslandLightForecastQ(ii,:)))=0;
    FYIslandLightForecastQ(ii,isnan(FYIslandLightForecastQ(ii,:)))=0;    
    
    % National and regional totals
    
    RegionNum = GXPReg(ii);     
    % National load at national peak
    ESNationalForecast(1,:) = ESNationalForecast(1,:) + ESForecastP(ii,:);
    SNationalForecast(1,:) = SNationalForecast(1,:) + SForecastP(ii,:);
    WNationalForecast(1,:) = WNationalForecast(1,:) + WForecastP(ii,:);
    FYNationalForecast(1,:) = FYNationalForecast(1,:) + FYForecastP(ii,:);
    % Region load at island peak
    ESRegionIslandPeakForecast(RegionNum,:) = ESRegionIslandPeakForecast(RegionNum,:) + ESIslandForecastP(ii,:); % do strmatch on names to avoid problems if ADMD and mapping list not identical
    SRegionIslandPeakForecast(RegionNum,:) = SRegionIslandPeakForecast(RegionNum,:) + SIslandForecastP(ii,:);
    WRegionIslandPeakForecast(RegionNum,:) = WRegionIslandPeakForecast(RegionNum,:) + WIslandForecastP(ii,:); 
    FYRegionIslandPeakForecast(RegionNum,:) = FYRegionIslandPeakForecast(RegionNum,:) + FYIslandForecastP(ii,:);
    % Region load at region peak
    ESRegionForecast(RegionNum,:) = ESRegionForecast(RegionNum,:) + ESRegionForecastP(ii,:);
    SRegionForecast(RegionNum,:) = SRegionForecast(RegionNum,:) + SRegionForecastP(ii,:);
    WRegionForecast(RegionNum,:) = WRegionForecast(RegionNum,:) + WRegionForecastP(ii,:);       
    FYRegionForecast(RegionNum,:) = FYRegionForecast(RegionNum,:) + FYRegionForecastP(ii,:);  
    % Region load at island minimum
    ESRegionIslandLightForecast(RegionNum,:) = ESRegionIslandLightForecast(RegionNum,:) + ESIslandLightForecastP(ii,:);
    SRegionIslandLightForecast(RegionNum,:) = SRegionIslandLightForecast(RegionNum,:) + SIslandLightForecastP(ii,:);
    WRegionIslandLightForecast(RegionNum,:) = WRegionIslandLightForecast(RegionNum,:) + (WIslandLightForecastP(ii,:));
    FYRegionIslandLightForecast(RegionNum,:) = FYRegionIslandLightForecast(RegionNum,:) + (FYIslandLightForecastP(ii,:));
    % Island load at island peak
    if char(ADMDGXPMapping(strmatch(ADMDnames(ii),ADMDGXPMapping(:,1),'exact'),3)) == char('N');
        ESIslandForecast(1,:)= ESIslandForecast(1,:)+(ESIslandForecastP(ii,:)); 
        SIslandForecast(1,:)= SIslandForecast(1,:)+(SIslandForecastP(ii,:));
        WIslandForecast(1,:)= WIslandForecast(1,:)+(WIslandForecastP(ii,:));
        FYIslandForecast(1,:)= FYIslandForecast(1,:)+(FYIslandForecastP(ii,:));
    else
        ESIslandForecast(2,:)= ESIslandForecast(2,:)+(ESIslandForecastP(ii,:)); 
        SIslandForecast(2,:)= SIslandForecast(2,:)+(SIslandForecastP(ii,:));
        WIslandForecast(2,:)= WIslandForecast(2,:)+(WIslandForecastP(ii,:));
        FYIslandForecast(2,:)= FYIslandForecast(2,:)+(FYIslandForecastP(ii,:));
    end;
    % Island load at island minimum
    if char(ADMDGXPMapping(strmatch(ADMDnames(ii),ADMDGXPMapping(:,1),'exact'),3)) == char('N');
        ESIslandLightForecast(1,:)= ESIslandLightForecast(1,:)+(ESIslandLightForecastP(ii,:)); 
        SIslandLightForecast(1,:)= SIslandLightForecast(1,:)+(SIslandLightForecastP(ii,:));
        WIslandLightForecast(1,:)= WIslandLightForecast(1,:)+(WIslandLightForecastP(ii,:));
        FYIslandLightForecast(1,:)= FYIslandLightForecast(1,:)+(FYIslandLightForecastP(ii,:));
    else
        ESIslandLightForecast(2,:)= ESIslandLightForecast(2,:)+(ESIslandLightForecastP(ii,:)); 
        SIslandLightForecast(2,:)= SIslandLightForecast(2,:)+(SIslandLightForecastP(ii,:));
        WIslandLightForecast(2,:)= WIslandLightForecast(2,:)+(WIslandLightForecastP(ii,:));
        FYIslandLightForecast(2,:)= FYIslandLightForecast(2,:)+(FYIslandLightForecastP(ii,:));
    end;    
    % Island load at national peak
    if char(ADMDGXPMapping(strmatch(ADMDnames(ii),ADMDGXPMapping(:,1),'exact'),3)) == char('N');
        ESIslandNationalForecast(1,:)= ESIslandNationalForecast(1,:)+(ESForecastP(ii,:)); 
        SIslandNationalForecast(1,:)= SIslandNationalForecast(1,:)+(SForecastP(ii,:));
        WIslandNationalForecast(1,:)= WIslandNationalForecast(1,:)+(WForecastP(ii,:));
        FYIslandNationalForecast(1,:)= FYIslandNationalForecast(1,:)+(FYForecastP(ii,:));
    else
        ESIslandNationalForecast(2,:)= ESIslandNationalForecast(2,:)+(ESForecastP(ii,:)); 
        SIslandNationalForecast(2,:)= SIslandNationalForecast(2,:)+(SForecastP(ii,:));
        WIslandNationalForecast(2,:)= WIslandNationalForecast(2,:)+(WForecastP(ii,:));
        FYIslandNationalForecast(2,:)= FYIslandNationalForecast(2,:)+(FYForecastP(ii,:)); 
    end;
end;

% Print forecasts in various formats

ny=YearStp-YearStt+1;
AllYears=BaseYear:YearStp;
yidx=find(AllYears==YearStt)-1;


% Save forecasts in dle format for DigSilent
% Name mappings are in the ADMDPSSEMapping variable in the PSSE_GXP_NameMapping.mat file

% Runs are split between North and South Island to produce separate files

PSSEname = ADMDPSSEMapping(:,1);
DLEname = ADMDPSSEMapping(:,2);

%%% remove this loop since now only 1 island
for IslandID = 1:2;
    if IslandID == 1; Island = 'N'; else Island = 'S'; end

    pd=cd;
%ak DigSilent loop commented out as not used
%%%    cd dles
%%%    for jj=1:ny
%%%        yri=yidx+jj;
%%%        fname=[[char(Island) 'I_ExtremeSummer'],num2str(AllYears(yri),'%4.0f'),'.dle'];
%%%        fidES=fopen(fname,'w');
%%%        fprintf(fidES,'%s \r\n','dole/dbupd level=1 langue=0');
%%%        fprintf(fidES,'%s \r\n','');
%%%        fprintf(fidES,'%s \r\n','** Load Dole from Matlab script MakeIDEV');
%%%        if ExportDataType == 1;
%%%            fprintf(fidES,'%s%s \r\n','** Island Extreme Summer Case, created ',datestr(now));
%%%        else 
%%%            if ExportDataType == 2;
%%%                fprintf(fidES,'%s%s \r\n','** Island Light Load Extreme Summer Case, created ',datestr(now));
%%%            else
%%%                fprintf(fidES,'%s%s \r\n','** Region Extreme Summer Case, created ',datestr(now));
%%%            end;
%%%        end;
%%%        fprintf(fidES,'%s \r\n','cls');
%%%        fprintf(fidES,'%s \r\n','set/def obj=ElmLod var=mode_inp,plini,qlini');
%%%        
%%%        fname=[[char(Island) 'I_Summer'],num2str(AllYears(yri),'%4.0f'),'.dle'];
%%%        fidS=fopen(fname,'w');
%%%        fprintf(fidS,'%s \r\n','dole/dbupd level=1 langue=0');
%%%        fprintf(fidS,'%s \r\n','');
%%%        fprintf(fidS,'%s \r\n','** Load Dole from Matlab script MakeIDEV');
%%%        if ExportDataType == 1;
%%%            fprintf(fidS,'%s%s \r\n','** Island Summer Case, created ',datestr(now));
%%%        else 
%%%            if ExportDataType == 2;
%%%                fprintf(fidS,'%s%s \r\n','** Island Light Load Summer Case, created ',datestr(now));
%%%            else
%%%                fprintf(fidS,'%s%s \r\n','** Region Summer Case, created ',datestr(now));
%%%            end;
%%%        end;
%%%        fprintf(fidS,'%s \r\n','cls');
%%%        fprintf(fidS,'%s \r\n','set/def obj=ElmLod var=mode_inp,plini,qlini');
%%%        
%%%        fname=[[char(Island) 'I_Winter'],num2str(AllYears(yri),'%4.0f'),'.dle'];
%%%        fidW=fopen(fname,'w');
%%%        fprintf(fidW,'%s \r\n','dole/dbupd level=1 langue=0');
%%%        fprintf(fidW,'%s \r\n','');
%%%        fprintf(fidW,'%s \r\n','** Load Dole from Matlab script MakeIDEV');
%%%        if ExportDataType == 1;
%%%            fprintf(fidW,'%s%s \r\n','** Island Winter Case, created ',datestr(now));
%%%        else 
%%%            if ExportDataType == 2;
%%%                fprintf(fidW,'%s%s \r\n','** Island Light Load Winter Case, created ',datestr(now));
%%%            else
%%%                fprintf(fidW,'%s%s \r\n','** Region Winter Case, created ',datestr(now));
%%%            end;
%%%        end;
%%%        fprintf(fidW,'%s \r\n','cls');
%%%        fprintf(fidW,'%s \r\n','set/def obj=ElmLod var=mode_inp,plini,qlini');
%%%        for ii=1:nGXP
%%%            idx=strmatch(ADMDnames(ii),PSSEname,'exact');
%%%            if (isempty(idx) | (size(idx,1) > 1)) | (char(ADMDIsland(ii)) ~= char(Island))
%%%                if (jj==1  & ~(char(ADMDIsland(ii)) ~= char(Island)))
%%%                    disp(['Problem matching names (DigSilent) ',num2str(size(idx,1)),' ',char(ADMDnames(ii)),' so ignored.' ]);
%%%                end
%%%            else
%%%                if ExportDataType == 1;
%%%                    oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(ESIslandForecastP(ii,yri),'%7.2f'),',',num2str(ESIslandForecastQ(ii,yri),'%7.2f')];
%%%                    fprintf(fidES,'%s\r\n',oneline);
%%%                    oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(SIslandForecastP(ii,yri),'%7.2f'),',',num2str(SIslandForecastQ(ii,yri),'%7.2f')];
%%%                    fprintf(fidS,'%s\r\n',oneline);
%%%                    oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(WIslandForecastP(ii,yri),'%7.2f'),',',num2str(WIslandForecastQ(ii,yri),'%7.2f')];
%%%                    fprintf(fidW,'%s\r\n',oneline);
%%%                else 
%%%                    if ExportDataType == 2;
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(ESIslandLightForecastP(ii,yri),'%7.2f'),',',num2str(ESIslandLightForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidES,'%s\r\n',oneline);
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(SIslandLightForecastP(ii,yri),'%7.2f'),',',num2str(SIslandLightForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidS,'%s\r\n',oneline);
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(WIslandLightForecastP(ii,yri),'%7.2f'),',',num2str(WIslandLightForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidW,'%s\r\n',oneline);
%%%                    else
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(ESRegionForecastP(ii,yri),'%7.2f'),',',num2str(ESRegionForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidES,'%s\r\n',oneline);
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(SRegionForecastP(ii,yri),'%7.2f'),',',num2str(SRegionForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidS,'%s\r\n',oneline);
%%%                        oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=PQ,',num2str(WRegionForecastP(ii,yri),'%7.2f'),',',num2str(WRegionForecastQ(ii,yri),'%7.2f')];
%%%                        fprintf(fidW,'%s\r\n',oneline);
%%%                    end;
%%%                end;               
%%%            end
%%%        end;      
%%%        fprintf(fidES,'%s \r\n','');
%%%        fprintf(fidES,'%s \r\n','set/def obj=ElmLod var=mode_inp');
%%%        fprintf(fidS,'%s \r\n','');
%%%        fprintf(fidS,'%s \r\n','set/def obj=ElmLod var=mode_inp');
%%%        fprintf(fidW,'%s \r\n','');
%%%        fprintf(fidW,'%s \r\n','set/def obj=ElmLod var=mode_inp');
%%%        for ii=1:nGXP
%%%            idx=strmatch(ADMDnames(ii),PSSEname,'exact');
%%%            if ~(isempty(idx) | (size(idx,1) > 1)) & (char(ADMDIsland(ii)) == char(Island))
%%%                oneline=['set obj=',char(DLEname(idx)),'.ElmLod val=DEF'];
%%%                fprintf(fidES,'%s\r\n',oneline);
%%%                fprintf(fidS,'%s\r\n',oneline);
%%%                fprintf(fidW,'%s\r\n',oneline);
%%%            end
%%%        end
%%%        
%%%        fprintf(fidES,'%s \r\n','');
%%%        status=fclose(fidES);
%%%        fprintf(fidS,'%s \r\n','');
%%%        status=fclose(fidS);
%%%        fprintf(fidW,'%s \r\n','');
%%%        status=fclose(fidW);
%%%    end
%%%    cd(pd)

    % Save forecasts in IDEV format
 
    % IDEV format requires PSS/E bus numbers to be known, so a PSS/E raw file
    % is read in to get a busname - bus number mapping
    
    if Island == 'N';PSSEFile=PSSE_NorthIsland_File; else PSSEFile=PSSE_SouthIsland_File; end
    
    PSSERead; % 2 island version

%ak NZ RAW file, not used yet
%    PSSEFile=PSSE_NZ_File; 
%    PSSEReadAll; % code for reading 1-island raw file
 
    pd=cd;
    cd idevs
    for jj=1:ny
        yri=yidx+jj;
        fname=[[char(Island) 'I_ExtremeSummer'],num2str(AllYears(yri),'%4.0f'),'.idv'];
        fidES=fopen(fname,'w');
        fname=[[char(Island) 'I_Summer'],num2str(AllYears(yri),'%4.0f'),'.idv'];
        fidS=fopen(fname,'w');
        fname=[[char(Island) 'I_Winter'],num2str(AllYears(yri),'%4.0f'),'.idv'];
        fidW=fopen(fname,'w');

        if ExportDataType == 1;
            fprintf(fidES,'%s \r\n','@!Text   ** Island Extreme Summer peak demand from Matlab script MakeIDEV ,,,,,,');
        else 
            if ExportDataType == 2;
                fprintf(fidES,'%s \r\n','@!Text   ** Light Load Extreme Summer demand from Matlab script MakeIDEV ,,,,,,');
            else
                fprintf(fidES,'%s \r\n','@!Text   ** Region Extreme Summer peak demand from Matlab script MakeIDEV ,,,,,,');
            end;
        end;      
        fprintf(fidES,'%s%s%s \r\n','Text   ** created ',datestr(now),',,,,,,');
        fprintf(fidES,'%s \r\n','');
        fprintf(fidES,'%s \r\n','RDCH,,,,,,');
        fprintf(fidES,'%s \r\n','1,,,,,,');
        fprintf(fidES,'%s \r\n','0,,,,,,');        
        
        if ExportDataType == 1;
            fprintf(fidS,'%s \r\n','@!Text   ** Island Summer peak demand from Matlab script MakeIDEV ,,,,,,');
        else 
            if ExportDataType == 2;
                fprintf(fidS,'%s \r\n','@!Text   ** Light Load Summer demand from Matlab script MakeIDEV ,,,,,,');
            else
                fprintf(fidS,'%s \r\n','@!Text   ** Region Summer peak demand from Matlab script MakeIDEV ,,,,,,');
            end;
        end;
        fprintf(fidS,'%s%s%s \r\n','Text   ** created ',datestr(now),',,,,,,');
        fprintf(fidS,'%s \r\n','');
        fprintf(fidS,'%s \r\n','RDCH,,,,,,');
        fprintf(fidS,'%s \r\n','1,,,,,,');
        fprintf(fidS,'%s \r\n','0,,,,,,');
        
        if ExportDataType == 1;
            fprintf(fidW,'%s \r\n','@!Text   ** Island Winter peak demand from Matlab script MakeIDEV ,,,,,,');
        else 
            if ExportDataType == 2;
                fprintf(fidW,'%s \r\n','@!Text   ** Light Load Winter demand from Matlab script MakeIDEV ,,,,,,');
            else
                fprintf(fidW,'%s \r\n','@!Text   ** Region Winter peak demand from Matlab script MakeIDEV ,,,,,,');
            end;
        end;
        fprintf(fidW,'%s%s%s \r\n','Text   ** created ',datestr(now),',,,,,,');
        fprintf(fidW,'%s \r\n','');
        fprintf(fidW,'%s \r\n','RDCH,,,,,,');
        fprintf(fidW,'%s \r\n','1,,,,,,');
        fprintf(fidW,'%s \r\n','0,,,,,,');
        
        for ii=1:nGXP
            idx=strmatch(ADMDPSSEMapping(ii,2),strtrim(PSSE.(Island).BUS.NAME),'exact');
            if isempty(idx) | (size(idx,1) > 1) | (char(ADMDIsland(ii)) ~= char(Island))
                if (jj==1 & (char(ADMDIsland(ii)) == char(Island)) & ~strcmp(char(ADMDPSSEMapping(ii,2)),char('ZERO'))) % Ignore the 'Zero' buses
                    disp(['Problem matching names ',num2str(size(idx,1)),' ',char(ADMDnames(ii)),' so ignored.' ]);
                end
            else
                if ExportDataType == 1;
                    oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(ESIslandForecastP(ii,yri),'%12.7f'),',',num2str(ESIslandForecastQ(ii,yri),'%12.7f')];
                    fprintf(fidES,'%s\r\n',oneline);
                    oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(SIslandForecastP(ii,yri),'%12.7f'),',',num2str(SIslandForecastQ(ii,yri),'%12.7f')];
                    fprintf(fidS,'%s\r\n',oneline);
                    oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(WIslandForecastP(ii,yri),'%12.7f'),',',num2str(WIslandForecastQ(ii,yri),'%12.7f')];
                    fprintf(fidW,'%s\r\n',oneline);
                else 
                    if ExportDataType == 2;
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(ESIslandLightForecastP(ii,yri),'%12.7f'),',',num2str(ESIslandLightForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidES,'%s\r\n',oneline);
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(SIslandLightForecastP(ii,yri),'%12.7f'),',',num2str(SIslandLightForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidS,'%s\r\n',oneline);
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(WIslandLightForecastP(ii,yri),'%12.7f'),',',num2str(WIslandLightForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidW,'%s\r\n',oneline);
                    else
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(ESRegionForecastP(ii,yri),'%12.7f'),',',num2str(ESRegionForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidES,'%s\r\n',oneline);
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(SRegionForecastP(ii,yri),'%12.7f'),',',num2str(SRegionForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidS,'%s\r\n',oneline);
                        oneline=[num2str(PSSE.(Island).BUS.I(idx),'%4.0f'),',,,,,',num2str(WRegionForecastP(ii,yri),'%12.7f'),',',num2str(WRegionForecastQ(ii,yri),'%12.7f')];
                        fprintf(fidW,'%s\r\n',oneline);
                    end;
                end;
            end;
        end;
        
        fprintf(fidES,'%s \r\n','Q,,,,,,');
        fprintf(fidES,'%s \r\n','@end,,,,,,');
        fprintf(fidES,'%s \r\n','');
        status=fclose(fidES);
        fprintf(fidS,'%s \r\n','Q,,,,,,');
        fprintf(fidS,'%s \r\n','@end,,,,,,');
        fprintf(fidS,'%s \r\n','');
        status=fclose(fidS);
        fprintf(fidW,'%s \r\n','Q,,,,,,');
        fprintf(fidW,'%s \r\n','@end,,,,,,');
        fprintf(fidW,'%s \r\n','');
        status=fclose(fidW);        
        
    end;
    cd(pd)
end;

% Save GXP forecasts in csv format

yearline=',';
PQline=',';
for ii=YearStt:YearStp
    yearline=[yearline,num2str(ii,'%5.0f'),',,,'];
    PQline=[PQline,'P,Q,,'];
end
fidES=fopen('ADMDExtremeSummer.csv','w');
fidS=fopen('ADMDSummer.csv','w');
fidW=fopen('ADMDWinter.csv','w');
fprintf(fidES,'%s \r\n',yearline);
fprintf(fidES,'%s \r\n',PQline);
fprintf(fidS,'%s \r\n',yearline);
fprintf(fidS,'%s \r\n',PQline);
fprintf(fidW,'%s \r\n',yearline);
fprintf(fidW,'%s \r\n',PQline);
for ii=1:nGXP
    ESoneline=[char(ADMDnames(ii)),','];
    Soneline=[char(ADMDnames(ii)),','];
    Woneline=[char(ADMDnames(ii)),','];
    for jj=1:ny
        ESoneline=[ESoneline,num2str(ESIslandForecastP(ii,yidx+jj),'%7.3f'),',',num2str(ESIslandForecastQ(ii,yidx+jj),'%7.3f'),',,'];
        Soneline=[Soneline,num2str(SIslandForecastP(ii,yidx+jj),'%7.3f'),',',num2str(SIslandForecastQ(ii,yidx+jj),'%7.3f'),',,'];
        Woneline=[Woneline,num2str(WIslandForecastP(ii,yidx+jj),'%7.3f'),',',num2str(WIslandForecastQ(ii,yidx+jj),'%7.3f'),',,'];
    end;
    fprintf(fidES,'%s \r\n',ESoneline);
    fprintf(fidS,'%s \r\n',Soneline);
    fprintf(fidW,'%s \r\n',Woneline);
end
status=fclose(fidES);
status=fclose(fidS);
status=fclose(fidW);

% Create .csv file for setting up Excel ADMD spreadsheet. The output does not include extreme summer which is assumed to be included in the summer period - see the BaseADMD script.

fidXL=fopen('ADMD2009Data.csv','w');
switchstate  = ['Switches - Embedded added back : ',int2str(AddBackEmbedded), '  Add new load : ',int2str(AdjustforNewLoads), '  Scale to Max Likelihood peaks : ',int2str(ScaletoMaxLikelihoodPeaks),'  Tiwai constant : ', int2str(KeepTiwaiPeakConstant)];
fprintf(fidXL,'%s%s%s \r\n','2009 ADMD Output',',,,',switchstate);
fprintf(fidXL,'%s \r\n','');
fprintf(fidXL,'%s','Bus Code,Bus Name,Bus No,Peak Region Power Factor,Peak Island Power Factor,Winter Island Power Factor,Summer Island Power Factor,Peak Region Diversity,Peak Island Diversity,Winter Island Diversity,Summer Island Diversity');
for BusYr = YearStt:YearStp
    fprintf(fidXL,'%s%4.0f',',',BusYr);
end
fprintf(fidXL,'%s \r\n',',Embedded Gen MW');
fprintf(fidXL,'%s \r\n','');
for reg = 1:nRegions; % run through each region
    fprintf(fidXL,'%s \r\n',Regions{reg}); % print region header
    GXPInd = strmatch(Regions{reg},ADMDGXPMapping(:,4)); % Get GXPs for region
    for gxpnum = 1:length(GXPInd); 
        BusCode = char(ADMDPSSEMapping(GXPInd(gxpnum),2));
        BusIsland = char(ADMDIsland(GXPInd(gxpnum)));
        if isempty(strmatch(BusCode,'ZERO')); 
            BusIdx = strmatch(ADMDPSSEMapping(GXPInd(gxpnum),2),strtrim(PSSE.(BusIsland).BUS.NAME),'exact');
            BusNo = PSSE.(BusIsland).BUS.I(BusIdx);        
            BusName = char(ADMDGXPMapping(GXPInd(gxpnum),5));
            BusFYRegionPF = FYRegionPF(GXPInd(gxpnum)); 
            BusFYIslandPF = FYIslandPF(GXPInd(gxpnum));
            BusWIslandPF = WIslandPF(GXPInd(gxpnum));
            BusSIslandPF = SIslandPF(GXPInd(gxpnum));
            BusWIslandDF = WIslandDF(GXPInd(gxpnum));
            BusSIslandDF = SIslandDF(GXPInd(gxpnum));
            BusFYRegionDF = FYRegionDF(GXPInd(gxpnum));
            BusFYIslandDF = FYIslandDF(GXPInd(gxpnum));
            BusFCPeaks = GXPForecastP(GXPInd(gxpnum),:);
            BusEmbedded = ADMDGXPExistingEmbeddedMW(GXPInd(gxpnum));
            fprintf(fidXL,'%s%s%s%s%6.4f%s%6.4f%s%6.4f%s%6.4f%s%6.4f%s%6.4f%s%6.4f%s%6.4f%s%6.4f',...
            BusCode,',',BusName,',',BusNo,',',BusFYRegionPF,',',BusFYIslandPF,',',BusWIslandPF,',',BusSIslandPF,',',BusFYRegionDF,',',BusFYIslandDF,',',BusWIslandDF,',',BusSIslandDF);
            for BusYr = 1:YearStp-YearStt+1
                fprintf(fidXL,'%s%6.2f',',',BusFCPeaks(BusYr));
            end            
            fprintf(fidXL, '%s%6.2f \r\n',',',BusEmbedded);
        end;
    end;
    fprintf(fidXL,'%s',',Region peak,,,,,,,,,');
    for BusYr = 1:YearStp-YearStt+1
        fprintf(fidXL,'%s%6.2f',',',FYRegionForecast(reg,BusYr));
    end
    fprintf(fidXL, '\r\n');
    fprintf(fidXL,'%s',',Region total at Island peak,,,,,,,,,');
    for BusYr = 1:YearStp-YearStt+1
        fprintf(fidXL,'%s%6.2f',',',FYRegionIslandPeakForecast(reg,BusYr));
    end
    fprintf(fidXL, '\r\n');
    fprintf(fidXL,'%s \r\n','');
end;
fprintf(fidXL,'%s',',North Island Total,,,,,,,,,');
for BusYr = 1:YearStp-YearStt+1
    fprintf(fidXL,'%s%6.2f',',',FYIslandForecast(1,BusYr));
end
fprintf(fidXL, '\r\n');
fprintf(fidXL,'%s',',South Island Total,,,,,,,,,');
for BusYr = 1:YearStp-YearStt+1
    fprintf(fidXL,'%s%6.2f',',',FYIslandForecast(2,BusYr));
end
fprintf(fidXL, '\r\n');

status = fclose(fidXL);

cd(originaldirectory)
warning on MATLAB:divideByZero;

disp('Done');
toc;






































